//
//  WorkoutManager.swift
//  TrackWorkoutsSampleProject
//
//  Created by Letizia Granata on 27/08/25.
//

// Core workout management class - handles all HealthKit interactions
import HealthKit
import Foundation

// Main class responsible for managing workout sessions and HealthKit integration
@MainActor
class WorkoutManager: NSObject, ObservableObject {
    
    @Published var currentState: WorkoutState = .idle
    @Published var metrics = WorkoutMetrics()
    
    private let healthStore = HKHealthStore()           // HealthKit database access
    private var workoutSession: HKWorkoutSession?       // Current workout session
    private var workoutBuilder: HKLiveWorkoutBuilder?   // Builds and saves workout data
    private var startDate: Date?                        // When the workout started
    private var elapsedTimeTimer: Timer?                // Timer for updating elapsed time
    
    override init() {
        super.init()
        requestHealthKitPermission()
    }
    
    // Request permission to read and write health data
    private func requestHealthKitPermission() {
        guard
            let hr = HKObjectType.quantityType(forIdentifier: .heartRate),
            let kcal = HKObjectType.quantityType(forIdentifier: .activeEnergyBurned),
            let dist = HKObjectType.quantityType(forIdentifier: .distanceWalkingRunning)
        else {
            return
        }
        
        let read: Set = [hr, kcal, dist, HKObjectType.workoutType()]
        let write: Set = [kcal, dist, HKObjectType.workoutType()]
        
        healthStore.requestAuthorization(toShare: write, read: read) { success, error in
            if let error {
                print("HealthKit auth error:", error.localizedDescription)
            } else {
                print(success ? "HealthKit access granted" : "HealthKit access denied")
            }
        }
    }
    
    // Starts a new workout session with the specified activity type
    func startWorkout(activity: HKWorkoutActivityType = .running, location: HKWorkoutSessionLocationType = .outdoor) async {
        
        // Reset metrics from any previous workout
        metrics.reset()
        currentState = .preparing
        
        do {
            // This defines what type of workout we're doing and where
            let workoutConfig = HKWorkoutConfiguration()
            workoutConfig.activityType = activity  // e.g., running, cycling, walking
            workoutConfig.locationType = location  // indoor, outdoor, or unknown
            
            // The session manages the overall workout lifecycle
            let workoutInstance = try HKWorkoutSession(healthStore: healthStore, configuration: workoutConfig)
            workoutInstance.delegate = self  // Set this class to receive session updates
            
            // The builder collects and saves all the workout data
            let dataBuilder = workoutInstance.associatedWorkoutBuilder()
            dataBuilder.delegate = self  // Set this class to receive data updates
            
            // This tells HealthKit to automatically collect sensor data during the workout
            dataBuilder.dataSource = HKLiveWorkoutDataSource(healthStore: healthStore,
                                                             workoutConfiguration: workoutConfig)
            
            // Store references so we can use them later
            workoutSession = workoutInstance
            workoutBuilder = dataBuilder
            
            // This activates sensors and external devices (like heart rate monitors)
            workoutInstance.prepare()
            
            await performCountdown()
            
            let workoutStartTime = Date()
            startDate = workoutStartTime
            workoutInstance.startActivity(with: workoutStartTime)
            
            try await dataBuilder.beginCollection(at: workoutStartTime)
            
            currentState = .active
            startElapsedTimeTimer()
            
            print("Workout started successfully")
            
        } catch {
            print("Failed to start workout: \(error.localizedDescription)")
            currentState = .idle
        }
    }
    
    // Shows a 3-second countdown to give sensors time to activate
    private func performCountdown() async {
        for i in (1...3).reversed() {
            print("Starting in \(i)...")
            // Wait 1 second between each countdown number
            try? await Task.sleep(nanoseconds: 1_000_000_000)
        }
        print("Workout started!")
    }
    
    // Starts a timer that updates the elapsed time every second
    private func startElapsedTimeTimer() {
        elapsedTimeTimer = Timer.scheduledTimer(withTimeInterval: 1.0, repeats: true) { _ in
            if let startTime = self.startDate {
                self.metrics.elapsedTime = Date().timeIntervalSince(startTime)
            }
        }
    }
    
    // Stops the elapsed time timer
    private func stopElapsedTimeTimer() {
        elapsedTimeTimer?.invalidate()
        elapsedTimeTimer = nil
    }
    
    // Temporarily pauses the workout (can be resumed later)
    func pauseWorkout() {
        workoutSession?.pause()
        currentState = .paused
        stopElapsedTimeTimer()
        print("Workout paused")
    }
    
    // Resumes a paused workout
    func resumeWorkout() {
        workoutSession?.resume()
        currentState = .active
        startElapsedTimeTimer()
        print("Workout resumed")
    }
    
    // Ends the current workout and saves it to HealthKit
    func endWorkout() {
        guard let session = workoutSession else { return }
        
        // Stop the workout activity - this allows final metrics to be collected
        session.stopActivity(with: Date())
        
        // Stop the UI timer
        stopElapsedTimeTimer()
        
        // Note: The actual saving happens in the session delegate method
        // when the session state changes to .stopped
        print("Ending workout...")
    }
    
    // Updates the UI with new health statistics
    private func updateMetrics(for statistics: HKStatistics) {
        // Check what type of health data this statistic represents
        switch statistics.quantityType {
            
            // Heart Rate Updates
        case HKQuantityType.quantityType(forIdentifier: .heartRate):
            // Get the most recent heart rate reading
            if let heartRateValue = statistics.mostRecentQuantity()?.doubleValue(for: .count().unitDivided(by: .minute())) {
                metrics.heartRate = heartRateValue
                metrics.isHeartRateAvailable = true
            }
            
            // Calories Updates
        case HKQuantityType.quantityType(forIdentifier: .activeEnergyBurned):
            // Get the total calories burned so far
            if let caloriesValue = statistics.sumQuantity()?.doubleValue(for: .kilocalorie()) {
                metrics.totalCalories = caloriesValue
            }
            
            // Distance Updates
        case HKQuantityType.quantityType(forIdentifier: .distanceWalkingRunning):
            // Get the total distance traveled so far
            if let distanceValue = statistics.sumQuantity()?.doubleValue(for: .meter()) {
                metrics.totalDistance = distanceValue
            }
            
        default:
            // Handle any other types of health data if needed
            break
        }
    }
}

// Extension to handle workout session state changes
extension WorkoutManager: HKWorkoutSessionDelegate {
    
    // Called whenever the workout session changes state (preparing, running, stopped, etc.)
    func workoutSession(_ workoutSession: HKWorkoutSession,
                        didChangeTo toState: HKWorkoutSessionState,
                        from fromState: HKWorkoutSessionState,
                        date: Date) {
        
        print("Workout session state changed: \(fromState.rawValue) → \(toState.rawValue)")
        
        // Handle the transition to stopped state (when workout ends)
        if toState == .stopped, let builder = workoutBuilder {
            Task {
                do {
                    try await builder.endCollection(at: date)
                    
                    let savedWorkout = try await builder.finishWorkout()
                    
                    workoutSession.end()
                    
                    await MainActor.run {
                        currentState = .finished
                        print("Workout saved to HealthKit: \(savedWorkout)")
                    }
                    
                } catch {
                    print("Failed to save workout: \(error.localizedDescription)")
                    await MainActor.run {
                        currentState = .idle
                    }
                }
            }
        }
    }
    
    // Called if there's an error with the workout session
    func workoutSession(_ workoutSession: HKWorkoutSession, didFailWithError error: Error) {
        print("Workout session failed: \(error.localizedDescription)")
        currentState = .idle
        stopElapsedTimeTimer()
    }
}

// Extension to handle incoming health data during the workout
extension WorkoutManager: HKLiveWorkoutBuilderDelegate {
    
    // Called whenever new health data is collected during the workout
    // This is where we get real-time updates for heart rate, calories, distance, etc.
    func workoutBuilder(_ workoutBuilder: HKLiveWorkoutBuilder,
                        didCollectDataOf collectedTypes: Set<HKSampleType>) {
        
        // Process each type of health data that was collected
        for sampleType in collectedTypes {
            // Only process quantity types (numeric health data)
            guard let quantityType = sampleType as? HKQuantityType else { continue }
            
            // Get the latest statistics for this type of data
            let statistics = workoutBuilder.statistics(for: quantityType)
            
            // Update our UI with the new data
            if let stats = statistics {
                updateMetrics(for: stats)
            }
        }
    }
    
    // Called when the workout builder collects a workout event
    // (This can be used for workout segments, markers, etc.)
    func workoutBuilderDidCollectEvent(_ workoutBuilder: HKLiveWorkoutBuilder) {
        // Handle workout events if needed
        // For example, lap markers, interval changes, etc.
    }
}
