//
//  WorkoutView.swift
//  TrackWorkoutsSampleProject
//
//  Created by Letizia Granata on 27/08/25.
//

// Main user interface for the workout app
import SwiftUI

struct WorkoutView: View {
    
    // Connect to our workout manager to get real-time updates
    @StateObject private var workoutManager = WorkoutManager()
    
    var body: some View {
        NavigationView {
            VStack(spacing: 30) {
                
                Text("Workout Tracker")
                    .font(.largeTitle)
                    .fontWeight(.bold)
                
                VStack(spacing: 20) {
                    
                    // Elapsed Time (always shown)
                    MetricCard(
                        title: "Duration",
                        value: formatTime(workoutManager.metrics.elapsedTime),
                        unit: "",
                        color: .blue
                    )
                    
                    // Heart Rate (only show if available or if workout is active)
                    if workoutManager.metrics.isHeartRateAvailable {
                        MetricCard(
                            title: "Heart Rate",
                            value: String(format: "%.0f", workoutManager.metrics.heartRate),
                            unit: "BPM",
                            color: .red
                        )
                    } else if workoutManager.currentState == .active {
                        // Show message about connecting heart rate monitor
                        VStack {
                            Image(systemName: "heart.slash")
                                .font(.title)
                                .foregroundColor(.orange)
                            Text("Connect a heart rate monitor")
                                .font(.caption)
                                .foregroundColor(.orange)
                        }
                        .padding()
                        .background(Color.orange.opacity(0.1))
                        .cornerRadius(10)
                    }
                    
                    // Calories Burned
                    MetricCard(
                        title: "Calories",
                        value: String(format: "%.0f", workoutManager.metrics.totalCalories),
                        unit: "kcal",
                        color: .orange
                    )
                    
                    // Distance Traveled
                    MetricCard(
                        title: "Distance",
                        value: String(format: "%.2f", workoutManager.metrics.totalDistance / 1000),
                        unit: "km",
                        color: .green
                    )
                }
                .padding(.horizontal)
                
                VStack(spacing: 15) {
                    
                    // Show different buttons based on current workout state
                    switch workoutManager.currentState {
                        
                    case .idle:
                        // Not currently working out - show start button
                        Button("Start Running Workout") {
                            Task {
                                await workoutManager.startWorkout(activity: .running, location: .outdoor)
                            }
                        }
                        .buttonStyle(StartButtonStyle())
                        
                    case .preparing:
                        // Workout is being prepared - show loading state
                        VStack {
                            ProgressView()
                                .scaleEffect(1.2)
                            Text("Preparing workout...")
                                .font(.headline)
                                .foregroundColor(.blue)
                        }
                        
                    case .active:
                        // Workout is running - show pause and end buttons
                        HStack(spacing: 20) {
                            Button("Pause") {
                                workoutManager.pauseWorkout()
                            }
                            .buttonStyle(PauseButtonStyle())
                            
                            Button("End Workout") {
                                workoutManager.endWorkout()
                            }
                            .buttonStyle(EndButtonStyle())
                        }
                        
                    case .paused:
                        // Workout is paused - show resume and end buttons
                        HStack(spacing: 20) {
                            Button("Resume") {
                                workoutManager.resumeWorkout()
                            }
                            .buttonStyle(ResumeButtonStyle())
                            
                            Button("End Workout") {
                                workoutManager.endWorkout()
                            }
                            .buttonStyle(EndButtonStyle())
                        }
                        
                    case .finished:
                        // Workout is complete - show summary and new workout button
                        VStack(spacing: 15) {
                            
                            // Success message
                            VStack {
                                Image(systemName: "checkmark.circle.fill")
                                    .font(.title)
                                    .foregroundColor(.green)
                                Text("Workout Complete!")
                                    .font(.title2)
                                    .fontWeight(.semibold)
                                    .foregroundColor(.green)
                                Text("Your workout has been saved to HealthKit")
                                    .font(.caption)
                                    .foregroundColor(.secondary)
                            }
                            
                            // Button to start a new workout
                            Button("Start New Workout") {
                                // Reset the workout manager for a new session
                                workoutManager.currentState = .idle
                                workoutManager.metrics.reset()
                            }
                            .buttonStyle(StartButtonStyle())
                        }
                    }
                }
                
                Spacer()
            }
            .padding()
            .navigationBarHidden(true)
        }
    }
        
    // Formats time interval into MM:SS format
    private func formatTime(_ timeInterval: TimeInterval) -> String {
        let minutes = Int(timeInterval) / 60
        let seconds = Int(timeInterval) % 60
        return String(format: "%02d:%02d", minutes, seconds)
    }
}
