//
//  BookEntity.swift
//  BooksShelfTransferable
//
//  Created by Giugliano Antonella on 07/03/25.
//
import AppIntents
import CoreTransferable

@AssistantEntity(schema: .books.book)
struct BookEntity: AppEntity {
    struct BookEntityQuery: EntityStringQuery {
        @MainActor
        func entities(for identifiers: [BookEntity.ID]) async throws -> [BookEntity] {
            let entries = try await DataManager.collectBooks(for: identifiers)
            return entries.map(\.entity)
        }
        
        func entities(matching string: String) async throws -> [BookEntity] {
            let entries = try await DataManager.collectBooks(matching: string)
            return entries.map(\.entity)
        }
        
        func suggestedEntities() async throws -> [BookEntity] {
            let entries = try await DataManager.collectAllBooks()
            return entries.map(\.entity)
        }
    }
    
    static var defaultQuery = BookEntityQuery()
    
    var displayRepresentation: DisplayRepresentation {
        DisplayRepresentation(stringLiteral: title ?? "No title")
    }
    
    let id: UUID
    
    var title: String?
    var seriesTitle: String?
    var author: String?
    var genre: String?
    var purchaseDate: Date?
    var contentType: BookContentType?
    var url: URL?
    
    init(item: Book) {
        id = item.id
        title = item.title
        author = item.author
        genre = item.genre?.rawValue
        contentType = item.contentType
        url = item.url
    }
}

extension BookEntity: Transferable{
    
    var plainText: String {
        let body =
        """
        📘 \(title ?? "Untitled")
        ✍️ Author: \(author ?? "Unknown")
        📚 Type: \( contentType?.rawValue ?? "—")
        🏷 Genre: \(genre ?? "—")
        """
        return body
    }
    
    static var transferRepresentation: some TransferRepresentation {
        
        ProxyRepresentation(exporting: \.plainText).visibility(.all)
    }
}
