//
//  AddBookView.swift
//  BooksShelfTransferable
//
//  Created by Giugliano Antonella on 04/03/25.
//
import SwiftUI
import UniformTypeIdentifiers

struct AddBookView: View {
    
    @Environment(NavigationManager.self) private var navigation
    @Environment(\.modelContext) private var modelContext
    @Environment(\.dismiss) var dismiss
    
    @State private var title: String = ""
    @State private var author: String = ""
    @State private var selectedContentType: BookContentType? = .eBook
    @State private var selectedGenre: Genre? = .unspecified
    @State private var read: Bool = false
    @State private var selectedURL: URL? = nil
    
    @State private var isDocumentPickerPresented = false
    
    var body: some View {
        
        NavigationStack {
            Form {
                Section {
                    TextField("Title", text: $title)
                    
                    TextField("Author", text: $author)
                }
                
                Section {
                    Picker("Genre", selection: $selectedGenre) {
                        ForEach(Genre.allCases) { genre in
                            Text(genre.rawValue.capitalized).tag(genre)
                        }
                    }
                }
                
                Section {
                    Toggle("Has been read", isOn: $read)
                }
                
                Section {
                    Picker("Type", selection: $selectedContentType) {
                        ForEach(BookContentType.allCases) { type in
                            Text(type.rawValue.capitalized).tag(type)
                        }
                    }
                }
                
                if selectedContentType == .eBook {
                    Section {
                        Button("Select the Book file") {
                            isDocumentPickerPresented = true
                        }
                        .fileImporter(
                            isPresented: $isDocumentPickerPresented,
                            allowedContentTypes: [.pdf],
                            onCompletion: handleFileSelection
                        )
                        
                        if let url = selectedURL {
                            Text(url.isFileURL ? url.lastPathComponent : url.absoluteString)
                        }
                    }
                }
            }
            
            .navigationTitle("New book")
            .navigationBarTitleDisplayMode(.inline)
            
            .toolbar {
                ToolbarItem(placement: .cancellationAction) {
                    Button("Dismiss") {
                        dismiss()
                    }
                }
                
                ToolbarItem(placement: .confirmationAction) {
                    Button("Save", action: saveBook)
                }
            }
        }
        
    }
    
    private func saveBook() {
        Task {
            let book = Book(
                title: title,
                author: author,
                genre: selectedGenre,
                contentType: selectedContentType,
                read: read,
                url: selectedURL
            )
            
            try await DataManager.createNewBook(book: book)
        }
        
        dismiss()
    }
    
    private func handleFileSelection(result: Result<URL, Error>) {
        switch result {
        case .success(let tempURL):
            selectFile(at: tempURL)
        case .failure(let error):
            print("File selection error: \(error)")
        }
    }
    
    fileprivate func selectFile(at url: URL) {
        let fileManager = FileManager.default
        
        do {
            // Get a persistent location in your Documents folder
            let docsURL = fileManager.urls(for: .documentDirectory, in: .userDomainMask).first!
            
            // Create a unique destination file URL
            let destinationURL = docsURL
                .appendingPathComponent(UUID().uuidString)
                .appendingPathExtension(url.pathExtension)
            
            _ = url.startAccessingSecurityScopedResource()
            try fileManager.copyItem(at: url, to: destinationURL)
            
            // Now use destinationURL persistently.
            self.selectedURL = destinationURL
            
            url.stopAccessingSecurityScopedResource()
        } catch {
            print("File import error: \(error.localizedDescription)")
        }
    }
}

#Preview {
    AddBookView()
        .environment(NavigationManager())
}
