//
//  BookDetailView.swift
//  BooksShelfTransferable
//
//  Created by Giugliano Antonella on 04/03/25.
//
import SwiftUI
import SwiftData
import WebKit

struct BookDetailView: View {
    
    @Environment(\.modelContext) private var modelContext
    
    var book: Book
    
    var body: some View {
        
        VStack(spacing: 16) {
            VStack(spacing: 8) {
                Text("\(book.title.capitalized)")
                    .font(.title)
                    .bold()
                    .fontDesign(.serif)
                    .multilineTextAlignment(.center)
                
                Text("\(book.author!.capitalized)")
                    .font(.title3)
            }
            
            HStack(spacing: 8) {
                Text("\(book.genre!.rawValue.capitalized)")
                Text("•")
                Text("\(book.contentType!.rawValue.capitalized)")
            }
            .font(.body)
            
            GroupBox("Status") {
                HStack {
                    Text("\(book.read ? "Already Read" : "Not read yet")")
                    Spacer()
                }
            }
            
            if let url = book.url {
                VStack(alignment: .leading) {
                    WebView(url: url)
                        .clipShape(.rect(cornerRadius: 10))
                        .overlay {
                            RoundedRectangle(cornerRadius: 10)
                                .stroke(lineWidth: 1)
                                .foregroundStyle(.gray.opacity(0.4))
                        }
                    
                    Text("\(book.url?.absoluteString ?? "No url available")")
                        .font(.footnote)
                }
            } else {
                Spacer()
            }
        }
        .padding()
    }
}

#Preview("Book") {
    @Previewable @State var testBook = Book(
        title: "A Game of Throne (A Song of Fire and Ice #1)",
        author: "George R.R. Martin",
        genre: .fantasy,
        contentType: .physicalBook,
        read: true,
        url: nil
    )
    
    BookDetailView(book: testBook)
        .environment(NavigationManager())
}

#Preview("eBook") {
    @Previewable @State var testBook = Book(
        title: "A Game of Throne (A Song of Fire and Ice #1)",
        author: "George R.R. Martin",
        genre: .fantasy,
        contentType: .eBook,
        read: true,
        url: URL(string: "https://www.goodreads.com/book/show/13496.A_Game_of_Thrones")!
    )
    
    BookDetailView(book: testBook)
        .environment(NavigationManager())
}
